import fdb
import json
import os
import sys

# Load Configuration
# Load Configuration
CONFIG_PATH = os.path.join(os.path.dirname(__file__), 'sync_config.json')

def load_config():
    if not os.path.exists(CONFIG_PATH):
        print(f"Error: {CONFIG_PATH} not found.")
        sys.exit(1)
    
    with open(CONFIG_PATH, 'r') as f:
        return json.load(f)

def test_connection(config):
    fb = config.get('firebird_settings', {})
    databases = config.get('databases', [])
    
    host = fb.get('host', '127.0.0.1')
    port = fb.get('port', 3050)
    user = fb.get('user', 'SYSDBA')
    password = fb.get('password', 'masterkey')
    
    print(f"--- Firebird Connection Test ---")
    print(f"Server: {host}:{port}")
    print(f"User: {user}")
    print(f"--------------------------------")
    
    for db in databases:
        name = db.get('name')
        path = db.get('path')
        enabled = db.get('enabled', False)
        
        if not enabled:
            print(f"[SKIP] {name}: Disabled in config.")
            continue
            
        print(f"[TEST] {name} ({path})...", end=' ')
        
        try:
            dsn = f"{host}/{port}:{path}"
            con = fdb.connect(dsn=dsn, user=user, password=password)
            
            # Basic sanity check
            cur = con.cursor()
            cur.execute("SELECT count(*) FROM RDB$RELATIONS")
            count = cur.fetchone()[0]
            
            print(f"SUCCESS! (System tables: {count})")
            con.close()
            
        except Exception as e:
            print(f"FAILED!")
            print(f"  Error: {e}")

    # --- Staging Connection Test ---
    print(f"\n--- Staging Connection Test ---")
    import requests
    staging = config.get('staging', {})
    url = staging.get('url')
    token = staging.get('api_token')
    
    if not url or "example.com" in url:
        print(f"[SKIP] Staging URL not configured ({url})")
    else:
        print(f"[TEST] {url}...", end=' ')
        try:
            # Try a status check on a dummy DB
            headers = {'Authorization': f"Bearer {token}"}
            resp = requests.get(f"{url}?cmd=status&db=TEST", headers=headers, timeout=5)
            if resp.status_code == 200:
                print(f"SUCCESS! (Response: {resp.status_code})")
            elif resp.status_code == 401:
                print(f"FAILED! (Unauthorized - Check Token)")
            else:
                print(f"FAILED! (Status: {resp.status_code})")
        except Exception as e:
            print(f"FAILED!")
            print(f"  Error: {e}")

if __name__ == "__main__":
    config = load_config()
    test_connection(config)
