import fdb
import json
import requests
import time
import sys
import os
import logging
from datetime import datetime, date
from decimal import Decimal

# --- Configuration ---
CONFIG_PATH = os.path.join(os.path.dirname(__file__), 'sync_config.json')
LOG_PATH = os.path.join(os.path.dirname(__file__), 'sync_agent.log')

# Logging Setup
logging.basicConfig(
    filename=LOG_PATH,
    level=logging.INFO,
    format='%(asctime)s [%(levelname)s] %(message)s'
)
console = logging.StreamHandler()
console.setLevel(logging.INFO)
logging.getLogger('').addHandler(console)

class DateTimeEncoder(json.JSONEncoder):
    """Encodes dates and decimals for JSON."""
    def default(self, obj):
        if isinstance(obj, (datetime, date)):
            return obj.isoformat()
        if isinstance(obj, Decimal):
            return float(obj)
        return super(DateTimeEncoder, self).default(obj)

def load_config():
    if not os.path.exists(CONFIG_PATH):
        logging.error(f"{CONFIG_PATH} not found.")
        sys.exit(1)
    with open(CONFIG_PATH, 'r') as f:
        return json.load(f)

def get_db_connection(config, db_name):
    fb = config.get('firebird_settings', {})
    databases = config.get('databases', [])
    
    # Find DB path
    db_path = next((d['path'] for d in databases if d['name'] == db_name), None)
    if not db_path:
        raise ValueError(f"Database {db_name} not found in config.")
        
    dsn = f"{fb.get('host')}/{fb.get('port')}:{db_path}"
    return fdb.connect(
        dsn=dsn, 
        user=fb.get('user'), 
        password=fb.get('password'),
        charset='UTF8'
    )

def get_remote_status(staging_url, token, db_name):
    """Ask Staging server for the Max ID of each table."""
    headers = {'Authorization': f"Bearer {token}"}
    try:
        resp = requests.get(f"{staging_url}?cmd=status&db={db_name}", headers=headers, timeout=10)
        resp.raise_for_status()
        return resp.json().get('max_ids', {})
    except Exception as e:
        logging.error(f"Failed to get status from Staging: {e}")
        return None

def push_data(staging_url, token, db_name, table, rows):
    """Push new rows to Staging."""
    headers = {
        'Authorization': f"Bearer {token}",
        'Content-Type': 'application/json'
    }
    payload = {
        'db': db_name,
        'table': table,
        'data': rows
    }
    
    # Use custom encoder for Dates/Decimals
    data_json = json.dumps(payload, cls=DateTimeEncoder)
    
    try:
        resp = requests.post(staging_url, data=data_json, headers=headers, timeout=30)
        resp.raise_for_status()
        return resp.json()
    except Exception as e:
        logging.error(f"Failed to push data for {table}: {e}")
        # Log response content if available for debugging
        if 'resp' in locals():
            logging.error(f"Response: {resp.text}")
        return None

def sync_table(con, staging_url, token, db_name, table, last_id):
    """Sync a single table based on ID > last_id."""
    cur = con.cursor()
    
    # Check if table exists (basic check) / Assuming schema exists as per plan
    
    # Fetch batch of new records
    # Limit to 1000 to avoid huge payloads
    sql = f"SELECT FIRST 1000 * FROM {table} WHERE ID > ? ORDER BY ID ASC"
    
    try:
        cur.execute(sql, (last_id,))
        rows = cur.fetchall()
        
        column_names = [d[0] for d in cur.description]
        
        if not rows:
            return 0
            
        # Convert to list of dicts
        data = []
        for row in rows:
            record = dict(zip(column_names, row))
            data.append(record)
            
        logging.info(f"  -> Found {len(data)} new records for {table} (Last ID: {last_id})")
        
        # Push 
        result = push_data(staging_url, token, db_name, table, data)
        
        if result and result.get('status') == 'ok':
            logging.info(f"  -> Successfully synced {result.get('updated')} rows.")
            
            # recursive check? or just wait for next loop?
            # If we got a full batch (1000), we could validly fetch again immediately.
            # For simplicity in this 'Pulse' logic, we'll wait for next loop unless critical.
            return len(data)
        else:
            logging.error(f"  -> Staging rejected update.")
            return 0
            
    except fdb.DatabaseError as e:
        logging.warning(f"  -> Table {table} access error (might not exist): {e}")
        return 0

def run_sync_loop():
    config = load_config()
    staging_conf = config.get('staging', {})
    URL = staging_conf.get('url')
    TOKEN = staging_conf.get('api_token')
    
    if "staging.example.com" in URL:
        logging.warning("Please configure the REAL Staging URL in sync_config.json")
        return

    logging.info("Starting Sync Agent (Pulse Mode)...")
    
    while True:
        logging.info("--- Sync Pulse ---")
        
        # Reload config in loop? (Optional, good for dynamic enable/disable)
        config = load_config()
        databases = config.get('databases', [])
        
        errors = 0
        
        for db in databases:
            if not db.get('enabled'):
                continue
                
            name = db.get('name')
            logging.info(f"Syncing DB: {name}")
            
            # 1. Get Staging Status
            remote_status = get_remote_status(URL, TOKEN, name)
            if remote_status is None:
                logging.error("Could not reach staging. Skipping DB.")
                errors += 1
                continue
                
            # 2. Connect to Local FDB
            try:
                con = get_db_connection(config, name)
                
                # 3. Iterate Tables
                # Focusing on the Transaction tables identified in Schema
                # In real prod, this list should be in config too.
                target_tables = [
                    'NL_TRANSACTIONS', 
                    'SL_SLTRAN', 
                    'STK_STOCKTRANSACTION',
                    'SL_SALESINVOICETRAN',
                    'PL_BILLTRAN'
                ]
                
                total_synced = 0
                for tbl in target_tables:
                    last_id = remote_status.get(tbl, 0)
                    count = sync_table(con, URL, TOKEN, name, tbl, last_id)
                    total_synced += count
                
                con.close()
                logging.info(f"Completed {name}. Total rows synced: {total_synced}")
                
            except Exception as e:
                logging.error(f"Error processing {name}: {e}")
                errors += 1
        
        # Sleep
        logging.info("Sleeping for 60 seconds...")
        time.sleep(60)

if __name__ == "__main__":
    run_sync_loop()
